const { pool } = require('../config/db');
const bcrypt = require('bcryptjs');

const createTables = async () => {
  const client = await pool.connect();
  try {
    await client.query('BEGIN');

    // --- ROLES ---
    await client.query(`
      CREATE TABLE IF NOT EXISTS roles (
        id SERIAL PRIMARY KEY,
        name VARCHAR(50) UNIQUE NOT NULL
      );
    `);

    const roles = ['admin', 'participant', 'member', 'company'];
    for (const role of roles) {
      await client.query(`
        INSERT INTO roles (name)
        VALUES ($1)
        ON CONFLICT (name) DO NOTHING;
      `, [role]);
    }

    // --- USERS ---
    await client.query(`
      CREATE TABLE IF NOT EXISTS users (
        id SERIAL PRIMARY KEY,
        username VARCHAR(100) UNIQUE NOT NULL,
        email VARCHAR(255) UNIQUE NOT NULL,
        password_hash VARCHAR(255) NOT NULL,
        role_id INTEGER REFERENCES roles(id),
        created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
      );
    `);

    // Add columns to users if they don't exist
    await client.query(`
      ALTER TABLE users 
      ADD COLUMN IF NOT EXISTS reset_token VARCHAR(255),
      ADD COLUMN IF NOT EXISTS reset_token_expiry TIMESTAMP WITH TIME ZONE,
      ADD COLUMN IF NOT EXISTS is_active BOOLEAN DEFAULT TRUE;
    `);

    // --- EVENTS ---
    await client.query(`
      CREATE TABLE IF NOT EXISTS events (
        id SERIAL PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        description TEXT,
        date TIMESTAMP WITH TIME ZONE NOT NULL,
        location VARCHAR(255),
        image_url VARCHAR(255),
        created_by INTEGER REFERENCES users(id),
        created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
      );
    `);

    // Add columns to events
    await client.query(`
      ALTER TABLE events 
      ADD COLUMN IF NOT EXISTS is_active BOOLEAN DEFAULT TRUE;
    `);

    // --- REGISTRATIONS ---
    // We need to handle the potential schema change for registrations
    // First, ensure the table exists (basic version if not exists)
    // But better to create it correctly if it doesn't exist.
    
    const registrationsExists = await client.query(`
        SELECT EXISTS (
            SELECT FROM information_schema.tables 
            WHERE table_schema = 'public' 
            AND table_name = 'registrations'
        );
    `);

    if (!registrationsExists.rows[0].exists) {
        // Create with NEW schema
        await client.query(`
            CREATE TABLE registrations (
                id SERIAL PRIMARY KEY,
                user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
                event_id INTEGER REFERENCES events(id) ON DELETE CASCADE,
                status VARCHAR(50) DEFAULT 'registered',
                registered_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
                details JSONB DEFAULT '{}'::jsonb
            );
        `);
    } else {
        // Table exists, ensure it has the correct columns and constraints
        
        // 1. Add details if missing
        await client.query(`
            ALTER TABLE registrations 
            ADD COLUMN IF NOT EXISTS details JSONB DEFAULT '{}'::jsonb;
        `);

        // 2. Check if 'id' column exists (indicator of new schema)
        const idColumnExists = await client.query(`
            SELECT EXISTS (
                SELECT FROM information_schema.columns 
                WHERE table_name = 'registrations' AND column_name = 'id'
            );
        `);

        if (!idColumnExists.rows[0].exists) {
            console.log('Migrating registrations table to new schema...');
            // Drop old PK
            await client.query(`
                ALTER TABLE registrations DROP CONSTRAINT IF EXISTS registrations_pkey;
            `);
            // Make user_id nullable
            await client.query(`
                ALTER TABLE registrations ALTER COLUMN user_id DROP NOT NULL;
            `);
            // Add ID column
            await client.query(`
                ALTER TABLE registrations ADD COLUMN id SERIAL PRIMARY KEY;
            `);
        }
    }

    // --- FORUM (POSTS, COMMENTS, LIKES) ---
    await client.query(`
      CREATE TABLE IF NOT EXISTS posts (
        id SERIAL PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        content TEXT NOT NULL,
        author_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
        created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
        likes_count INTEGER DEFAULT 0
      );
    `);

    await client.query(`
      CREATE TABLE IF NOT EXISTS comments (
        id SERIAL PRIMARY KEY,
        post_id INTEGER REFERENCES posts(id) ON DELETE CASCADE,
        author_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
        content TEXT NOT NULL,
        created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
      );
    `);

    await client.query(`
      CREATE TABLE IF NOT EXISTS post_likes (
        user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
        post_id INTEGER REFERENCES posts(id) ON DELETE CASCADE,
        PRIMARY KEY (user_id, post_id)
      );
    `);

    // --- AUDIT LOGS ---
    await client.query(`
      CREATE TABLE IF NOT EXISTS audit_logs (
        id SERIAL PRIMARY KEY,
        user_id INTEGER REFERENCES users(id) ON DELETE SET NULL,
        action VARCHAR(50) NOT NULL,
        details JSONB,
        ip_address VARCHAR(45),
        created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
      );
    `);

    // --- DEFAULT ADMIN ---
    const adminEmail = 'admin@aeer-ua.com';
    const adminPassword = 'admin123_secure_password'; // Change this in production!
    const adminUsername = 'Admin';

    const adminCheck = await client.query('SELECT * FROM users WHERE email = $1', [adminEmail]);

    if (adminCheck.rows.length === 0) {
      const salt = await bcrypt.genSalt(10);
      const hashedPassword = await bcrypt.hash(adminPassword, salt);
      
      const roleResult = await client.query("SELECT id FROM roles WHERE name = 'admin'");
      const adminRoleId = roleResult.rows[0].id;

      await client.query(`
        INSERT INTO users (username, email, password_hash, role_id)
        VALUES ($1, $2, $3, $4)
      `, [adminUsername, adminEmail, hashedPassword, adminRoleId]);
      
      console.log(`Default admin created: ${adminEmail}`);
    } else {
      console.log('Default admin already exists');
    }

    await client.query('COMMIT');
    console.log('Database initialized and verified successfully');
  } catch (e) {
    await client.query('ROLLBACK');
    console.error('Error initializing database', e);
  } finally {
    client.release();
    pool.end();
  }
};

createTables();
