const db = require('../config/db');
const bcrypt = require('bcryptjs');

const logAudit = async (userId, action, details, req) => {
  try {
    const ip = req.headers['x-forwarded-for'] || req.socket.remoteAddress;
    await db.query(
      'INSERT INTO audit_logs (user_id, action, details, ip_address) VALUES ($1, $2, $3, $4)',
      [userId, action, JSON.stringify(details), ip]
    );
  } catch (e) {
    console.error('Audit log error:', e);
  }
};

const getUsers = async (req, res, next) => {
  try {
    const result = await db.query(`
      SELECT u.id, u.username, u.email, u.is_active, u.created_at, r.name as role_name, r.id as role_id
      FROM users u 
      JOIN roles r ON u.role_id = r.id
      ORDER BY u.created_at DESC
    `);
    res.json(result.rows);
  } catch (error) {
    next(error);
  }
};

const getRoles = async (req, res, next) => {
  try {
    const result = await db.query('SELECT * FROM roles ORDER BY id');
    res.json(result.rows);
  } catch (error) {
    next(error);
  }
};

const createUser = async (req, res, next) => {
  const { username, email, password, role_id } = req.body;
  const adminId = req.user.id;

  try {
    // Check if user exists
    const userCheck = await db.query('SELECT * FROM users WHERE email = $1', [email]);
    if (userCheck.rows.length > 0) {
      return res.status(400).json({ message: 'User already exists' });
    }

    const salt = await bcrypt.genSalt(10);
    const hashedPassword = await bcrypt.hash(password, salt);

    const newUser = await db.query(
      'INSERT INTO users (username, email, password_hash, role_id, is_active) VALUES ($1, $2, $3, $4, true) RETURNING id, username, email',
      [username, email, hashedPassword, role_id]
    );

    await logAudit(adminId, 'CREATE_USER', { new_user_id: newUser.rows[0].id, email }, req);

    res.status(201).json({ message: 'User created successfully', user: newUser.rows[0] });
  } catch (error) {
    next(error);
  }
};

const updateUser = async (req, res, next) => {
  const { id } = req.params;
  const { username, email, role_id, password } = req.body;
  const adminId = req.user.id;

  try {
    let query = 'UPDATE users SET username = $1, email = $2, role_id = $3';
    let params = [username, email, role_id];
    let paramIndex = 4;

    if (password) {
      const salt = await bcrypt.genSalt(10);
      const hashedPassword = await bcrypt.hash(password, salt);
      query += `, password_hash = $${paramIndex}`;
      params.push(hashedPassword);
      paramIndex++;
    }

    query += ` WHERE id = $${paramIndex} RETURNING id, username, email`;
    params.push(id);

    const result = await db.query(query, params);

    if (result.rows.length === 0) {
      return res.status(404).json({ message: 'User not found' });
    }

    await logAudit(adminId, 'UPDATE_USER', { target_user_id: id, updates: { username, email, role_id } }, req);

    res.json({ message: 'User updated successfully', user: result.rows[0] });
  } catch (error) {
    next(error);
  }
};

const deleteUser = async (req, res, next) => {
  const { id } = req.params;
  const adminId = req.user.id;

  try {
    if (parseInt(id) === adminId) {
      return res.status(400).json({ message: 'Cannot delete yourself' });
    }

    await db.query('BEGIN');

    // 1. Reassign events created by this user to the admin
    await db.query('UPDATE events SET created_by = $1 WHERE created_by = $2', [adminId, id]);

    // 2. Handle Forum Posts & Comments (Set author to NULL to preserve content)
    // Note: This assumes author_id is nullable. If not, we might need to delete them.
    await db.query('UPDATE posts SET author_id = NULL WHERE author_id = $1', [id]);
    await db.query('UPDATE comments SET author_id = NULL WHERE author_id = $1', [id]);

    // 3. Remove likes
    await db.query('DELETE FROM post_likes WHERE user_id = $1', [id]);

    // 4. Unlink registrations (keep the record, but remove user link)
    await db.query('UPDATE registrations SET user_id = NULL WHERE user_id = $1', [id]);

    // 5. Delete the user
    await db.query('DELETE FROM users WHERE id = $1', [id]);
    
    await logAudit(adminId, 'DELETE_USER', { target_user_id: id }, req);

    await db.query('COMMIT');

    res.json({ message: 'User deleted successfully' });
  } catch (error) {
    await db.query('ROLLBACK');
    next(error);
  }
};

const toggleUserStatus = async (req, res, next) => {
  const { id } = req.params;
  const { is_active } = req.body;
  const adminId = req.user.id;

  try {
    if (parseInt(id) === adminId) {
      return res.status(400).json({ message: 'Cannot change your own status' });
    }

    await db.query('UPDATE users SET is_active = $1 WHERE id = $2', [is_active, id]);
    await logAudit(adminId, 'TOGGLE_STATUS', { target_user_id: id, new_status: is_active }, req);

    res.json({ message: 'User status updated' });
  } catch (error) {
    next(error);
  }
};

const getAuditLogs = async (req, res, next) => {
  try {
    const result = await db.query(`
      SELECT a.*, u.username as admin_name 
      FROM audit_logs a 
      LEFT JOIN users u ON a.user_id = u.id 
      ORDER BY a.created_at DESC 
      LIMIT 100
    `);
    res.json(result.rows);
  } catch (error) {
    next(error);
  }
};

module.exports = { getUsers, getRoles, createUser, updateUser, deleteUser, toggleUserStatus, getAuditLogs };
