const db = require('../config/db');

const getAllPosts = async (req, res, next) => {
  try {
    const result = await db.query(`
      SELECT 
        p.*, 
        u.username as author_name, 
        u.role_id,
        (SELECT COUNT(*)::int FROM comments c WHERE c.post_id = p.id) as comments_count,
        (SELECT COUNT(*)::int FROM post_likes pl WHERE pl.post_id = p.id) as likes_count,
        EXISTS(SELECT 1 FROM post_likes pl WHERE pl.post_id = p.id AND pl.user_id = $1) as is_liked
      FROM posts p
      JOIN users u ON p.author_id = u.id
      ORDER BY p.created_at DESC
    `, [req.user ? req.user.id : null]);
    res.json(result.rows);
  } catch (error) {
    next(error);
  }
};

const createPost = async (req, res, next) => {
  const { title, content } = req.body;
  const authorId = req.user.id;

  try {
    const result = await db.query(
      'INSERT INTO posts (title, content, author_id) VALUES ($1, $2, $3) RETURNING *',
      [title, content, authorId]
    );
    const newPost = result.rows[0];
    // Fetch author details to return complete object
    const authorResult = await db.query('SELECT username FROM users WHERE id = $1', [authorId]);
    newPost.author_name = authorResult.rows[0].username;
    newPost.comments_count = 0;
    newPost.likes_count = 0;
    newPost.is_liked = false;
    
    res.status(201).json(newPost);
  } catch (error) {
    next(error);
  }
};

const updatePost = async (req, res, next) => {
  const { id } = req.params;
  const { title, content } = req.body;
  const userId = req.user.id;
  const userRole = req.user.role; // Assuming role name is available, or check role_id

  try {
    // Check ownership or admin
    const postResult = await db.query('SELECT * FROM posts WHERE id = $1', [id]);
    if (postResult.rows.length === 0) return res.status(404).json({ message: 'Post not found' });
    
    const post = postResult.rows[0];
    
    // Check if user is admin (need to fetch role name if not in token, assuming token has role)
    // For now, let's assume req.user.role is populated by middleware or we check DB
    // Let's check DB for role if needed, but usually middleware handles it.
    // Assuming authorizeRole middleware is used, but here we need dynamic check (owner OR admin)
    
    // Let's fetch user role from DB to be safe if not in req.user
    const userResult = await db.query('SELECT r.name FROM users u JOIN roles r ON u.role_id = r.id WHERE u.id = $1', [userId]);
    const roleName = userResult.rows[0].name;

    if (post.author_id !== userId && roleName !== 'admin') {
      return res.status(403).json({ message: 'Not authorized' });
    }

    const result = await db.query(
      'UPDATE posts SET title = $1, content = $2, updated_at = CURRENT_TIMESTAMP WHERE id = $3 RETURNING *',
      [title, content, id]
    );
    res.json(result.rows[0]);
  } catch (error) {
    next(error);
  }
};

const deletePost = async (req, res, next) => {
  const { id } = req.params;
  const userId = req.user.id;

  try {
    const postResult = await db.query('SELECT * FROM posts WHERE id = $1', [id]);
    if (postResult.rows.length === 0) return res.status(404).json({ message: 'Post not found' });
    const post = postResult.rows[0];

    const userResult = await db.query('SELECT r.name FROM users u JOIN roles r ON u.role_id = r.id WHERE u.id = $1', [userId]);
    const roleName = userResult.rows[0].name;

    if (post.author_id !== userId && roleName !== 'admin') {
      return res.status(403).json({ message: 'Not authorized' });
    }

    await db.query('DELETE FROM posts WHERE id = $1', [id]);
    res.json({ message: 'Post deleted' });
  } catch (error) {
    next(error);
  }
};

const getPostComments = async (req, res, next) => {
  const { id } = req.params;
  try {
    const result = await db.query(`
      SELECT c.*, u.username as author_name 
      FROM comments c
      JOIN users u ON c.author_id = u.id
      WHERE c.post_id = $1
      ORDER BY c.created_at ASC
    `, [id]);
    res.json(result.rows);
  } catch (error) {
    next(error);
  }
};

const createComment = async (req, res, next) => {
  const { id } = req.params; // post_id
  const { content } = req.body;
  const authorId = req.user.id;

  try {
    const result = await db.query(
      'INSERT INTO comments (post_id, author_id, content) VALUES ($1, $2, $3) RETURNING *',
      [id, authorId, content]
    );
    const newComment = result.rows[0];
    const authorResult = await db.query('SELECT username FROM users WHERE id = $1', [authorId]);
    newComment.author_name = authorResult.rows[0].username;
    
    res.status(201).json(newComment);
  } catch (error) {
    next(error);
  }
};

const deleteComment = async (req, res, next) => {
  const { id, commentId } = req.params;
  const userId = req.user.id;

  try {
    const commentResult = await db.query('SELECT * FROM comments WHERE id = $1', [commentId]);
    if (commentResult.rows.length === 0) return res.status(404).json({ message: 'Comment not found' });
    const comment = commentResult.rows[0];

    const userResult = await db.query('SELECT r.name FROM users u JOIN roles r ON u.role_id = r.id WHERE u.id = $1', [userId]);
    const roleName = userResult.rows[0].name;

    if (comment.author_id !== userId && roleName !== 'admin') {
      return res.status(403).json({ message: 'Not authorized' });
    }

    await db.query('DELETE FROM comments WHERE id = $1', [commentId]);
    res.json({ message: 'Comment deleted' });
  } catch (error) {
    next(error);
  }
};

const toggleLike = async (req, res, next) => {
  const { id } = req.params;
  const userId = req.user.id;

  try {
    const checkLike = await db.query('SELECT * FROM post_likes WHERE post_id = $1 AND user_id = $2', [id, userId]);
    
    if (checkLike.rows.length > 0) {
      // Unlike
      await db.query('DELETE FROM post_likes WHERE post_id = $1 AND user_id = $2', [id, userId]);
      await db.query('UPDATE posts SET likes_count = likes_count - 1 WHERE id = $1', [id]);
      res.json({ liked: false });
    } else {
      // Like
      await db.query('INSERT INTO post_likes (post_id, user_id) VALUES ($1, $2)', [id, userId]);
      await db.query('UPDATE posts SET likes_count = likes_count + 1 WHERE id = $1', [id]);
      res.json({ liked: true });
    }
  } catch (error) {
    next(error);
  }
};

module.exports = {
  getAllPosts,
  createPost,
  updatePost,
  deletePost,
  getPostComments,
  createComment,
  deleteComment,
  toggleLike
};
